<?php
/**
 * Acumbamail Signup Forms - Main file
 *
 * @category   	Module
 * @author     	Acumbamail <jesus.botella@acumbamail.com>
 * @copyright  	2025 Acumbamail
 * @version   	1.2
 * @link       	http://www.acumbamail.com/
 */

// Security
if (!defined('_PS_VERSION_'))
    exit;

require_once(__DIR__ . '/api/acumbamail.class.php');

class Acumbamail extends Module {

    public function __construct()
    {
        $this->author = 'Acumbamail';
        $this->name = 'acumbamail';
        $this->version = '1.3';

        $this->ps_versions_compliancy['min'] = '1.5';
        $this->ps_versions_compliancy['max'] = '8.99.99';
        $this->need_instance = 0;

        parent::__construct();

        // Name in the modules list
        $this->displayName = $this->l('Acumbamail Signup Forms');
        $this->description = $this->l('Incluye fácilmente un formulario para que los usuarios de tu tienda se suscriban a tu newsletter de Acumbamail');
        $this->confirmUninstall = $this->l('¿Estás seguro de que deseas borrar este módulo?');

        $this->api = new AcumbamailAPI(0, Configuration::get('ACMB_AUTH_TOKEN'));
    }

    function hookLeftColumn($params) {
        if (Configuration::get('ACMB_FORM_TO_SHOW') != '') {
            $form_details = $this->api->getFormDetails(Configuration::get('ACMB_FORM_TO_SHOW'));
            $this->context->smarty->assign('acmb_form_details', $form_details);
        }

        return $this->display(__FILE__, 'views/templates/front/frontend.tpl');
    }

    function hookRightColumn($params) {
        return $this->hookLeftColumn($params);
    }

    function hookFooter($params) {
        return $this->hookLeftColumn($params);
    }

    public function hookActionValidateOrder($params) {
        $list_id = Configuration::get('ACMB_CLIENTS_LIST');
        $active_subscribers_status = 0;
        $subscribers = $this->api->getSubscribers($list_id, $active_subscribers_status);
        $already_subscribed = false;
        foreach (array_keys($subscribers) as $subscriber_key) {
            if ($subscribers[$subscriber_key]['email'] === $params['customer']->email) {
                $already_subscribed = true;
            }
        }

        $wants_subscription = $params['customer']->newsletter;

        if (!$already_subscribed && $wants_subscription) {
            $subscriber_fields = array();
            $subscriber_fields['email'] = $params['customer']->email;
            $this->api->addSubscriber($list_id, $subscriber_fields);
        }
    }

    public function install() {
        if (!parent::install() || !$this->registerHook('displayLeftColumn') || !$this->registerHook('actionValidateOrder'))
            return false;
        return true;
    }

    public function uninstall(){
        $this->flush_preferences();

        if (!parent::uninstall())
            return false;
        return true;
    }

    public function getContent() {
        $output = '<h2>'.$this->displayName.'</h2>';

        if (isset($errors) && count($errors)) {
            $output .= $this->displayError(implode('<br />', $errors));
        }

        //Submit del primer formulario (Configuración de Datos de Acumbamail)
        if (Tools::isSubmit('submit_form') || (Tools::isSubmit('ajax') && Tools::getValue('action') == 'prepare_sync_list'))
        {
            $this->formToSetAuthTokenSubmit();
            $output .= $this->formToShowInShopSubmit();
            $output .= $this->formToSubscribeClientsSubmit();
            $output .= $this->ajaxProcessPrepareSyncList();
        }

        if (Tools::isSubmit('ajax') && Tools::getValue('action') == 'sync_batch') {
            $this->ajaxProcessSyncBatch();
        }

        if (Tools::isSubmit('reset_form')) {
            $this->flush_preferences();
        }

        if (Tools::isSubmit('reset_sync_process')) {
            $this->flush_sync_process();
            $output .= $this->displayConfirmation($this->l('Proceso concluido. Ya puede volver a iniciar la sincronización.'));
        }

        return $output.$this->displayForm();
    }

    public function displayForm()
    {

        if(isset($errors))
            $this->context->smarty->assign('errors', $errors);
        else
            $errors = array();

        if (Configuration::get('ACMB_AUTH_TOKEN') != '') {
            $query_lists = $this->api->getLists();

            if ($query_lists != '') {
                $query_forms = $this->api->getForms(Configuration::get('ACMB_SINGLE_LIST'));
                $this->context->smarty->assign('listas', $query_lists);
                $this->context->smarty->assign('forms', $query_forms);
            }
            else {
                array_push($errors, $this->l("Token de Autenticación incorrecto: ") . Configuration::get('ACMB_AUTH_TOKEN'));
                $this->flush_preferences();
            }
        }

        if(Configuration::get("ACMB_SINGLE_LIST")!='' and Configuration::get("ACMB_FORM_TO_SHOW") != '') {
            $this->context->smarty->assign('acmb_form_details', $this->api->getFormDetails(Configuration::get('ACMB_FORM_TO_SHOW')));
        }

        $this->context->smarty->assign('request_uri', Tools::safeOutput($_SERVER['REQUEST_URI']));
        $this->context->smarty->assign('path', $this->_path);
        $this->context->smarty->assign('submitName', 'submit'.ucfirst($this->name));
        $this->context->smarty->assign('errors', $errors);
        $this->context->smarty->assign('acmb_auth_token', Configuration::get('ACMB_AUTH_TOKEN'));
        $this->context->smarty->assign('acmb_single_list', Configuration::get('ACMB_SINGLE_LIST'));
        $this->context->smarty->assign('acmb_form_to_show', Configuration::get('ACMB_FORM_TO_SHOW'));

        $selected_list["single"] = Configuration::get('ACMB_SINGLE_LIST');
        $this->context->smarty->assign('selected_list', $selected_list);

        $buyers_list["single"] = Configuration::get('ACMB_CLIENTS_LIST');
        $this->context->smarty->assign('buyers_list', $buyers_list);

        $selected_form["single"] = Configuration::get("ACMB_FORM_TO_SHOW");
        $this->context->smarty->assign('selected_form', $selected_form);

        $sync_in_process= Configuration::get("ACMB_SYNC_IN_PROCESS");
        $this->context->smarty->assign('sync_in_process', $sync_in_process);

        // You can return html, but I prefer this new version: use smarty in admin, :)
        return $this->display(__FILE__, 'views/templates/admin/configure.tpl');
    }

    private function formToSetAuthTokenSubmit() {
        $acumba_auth_token = Tools::getValue('acmb_auth_token');
        if($acumba_auth_token != '') {
            Configuration::updateValue('ACMB_AUTH_TOKEN',$acumba_auth_token);
            $this->api->setAuthToken($acumba_auth_token);
        }
        else {
            $this->flush_preferences();
        }
    }

    private function formToShowInShopSubmit() {
        $output_string = '';

        if(Tools::getValue('list-general')!='') {
            $single_list = Tools::getValue('list-general');
            Configuration::updateValue('ACMB_SINGLE_LIST', $single_list);
            $form_to_show = Tools::getValue('forms-general');
            if ($form_to_show != '') {
                Configuration::updateValue("ACMB_FORM_TO_SHOW", $form_to_show);
                $output_string .= $this->displayConfirmation($this->l('El formulario será mostrado en su tienda'));
            }
        }
        else {
            Configuration::deleteByName("ACMB_SINGLE_LIST");
            Configuration::deleteByName("ACMB_FORM_TO_SHOW");
        }

        return $output_string;
    }

    private function formToSubscribeClientsSubmit() {
        $output_string = '';
        $client_subscription_list = Tools::getValue('list-subscription');
        if($client_subscription_list != '') {
            Configuration::updateValue("ACMB_CLIENTS_LIST", $client_subscription_list);
            $output_string .= $this->displayConfirmation($this->l('Sus compradores se suscribirán automáticamente a la lista'));
        }
        else {
            Configuration::deleteByName("ACMB_CLIENTS_LIST");
        }
        return $output_string;
    }

    private function getSubscriptionsDifferences($client_subscription_list) {

        $this->createTempDb();

        $newsletter_subscriptions = $this->getNewsletterSubscriptions();
        $acumbamail_subscriptions = $this->api->getSubscribers($client_subscription_list, 0);
        $differences = array();
        foreach ($newsletter_subscriptions as $newsletter) {
            $found = false;
            if ($acumbamail_subscriptions) {
                foreach ($acumbamail_subscriptions as $acumbamail) {
                    if ($acumbamail['email'] == $newsletter['email']) {
                        $found = true;
                    }
                }
            }

            if (!$found) {
                array_push($differences, $newsletter);

                try {
                    Db::getInstance()->insert('acmb_subscriptions', [
                        'email' => pSQL($newsletter['email']),
                        'data' => pSQL(json_encode($newsletter)),
                        'synced' => 0,
                    ]);
                } catch(Exception $e){
                    error_log($e->getMessage());
                }
            }
        }
        return $differences;
    }

    private function getNewsletterSubscriptions() {
        $sql = new DbQuery();
        $sql->select('email');
        $sql->from('customer', 'c');
        $sql->where('c.newsletter=1');
        try {
            $newsletter_subscriptions = Db::getInstance()->executeS($sql);
        }
        catch (Exception $e) {
            $newsletter_subscriptions = array();
        }

        return $newsletter_subscriptions;
    }

    private function flush_preferences() {
        //Eliminamos cadenas de configuración
        Configuration::deleteByName("ACMB_AUTH_TOKEN");
        Configuration::deleteByName("ACMB_SINGLE_LIST");
        Configuration::deleteByName("ACMB_CLIENTS_LIST");
        Configuration::deleteByName("ACMB_FORM_TO_SHOW");
        $this->flush_sync_process();
    }

    public function ajaxProcessPrepareSyncList()
    {
        try {
            $client_subscription_list = Tools::getValue('list-subscription');
            if($client_subscription_list != '') {
                if (Tools::getValue('sync_newsletter_clients')) {
                    $difference = $this->getSubscriptionsDifferences($client_subscription_list);
                    die(json_encode([
                        'success' => true,
                        'total' => count($difference),
                        'client_subscription_list' => $client_subscription_list
                    ]));
                }
            }
            else {
                Configuration::deleteByName("ACMB_CLIENTS_LIST");
            }

        } catch (Exception $e) {
            error_log($e->getMessage());
            die(json_encode(['error' => $e->getMessage()]));
        }
    }

    public function ajaxProcessSyncBatch()
    {
        try {
            $offset = (int) Tools::getValue('offset', 0);
            $client_subscription_list = Tools::getValue('list_subscription');
            $limit = 400;

            $rows = Db::getInstance()->executeS("
                SELECT id, email, data
                FROM "._DB_PREFIX_."acmb_subscriptions
                WHERE synced = 0
                ORDER BY id ASC
                LIMIT $limit
            ");

            $list_subscriber = array();
            if ($rows != FALSE) {
                $processed = 0;
                $error = "";
                $detail_error = "";
                $list_id = "";
                $i = 0;
                $total = count($rows);
                Configuration::updateValue('ACMB_SYNC_IN_PROCESS',TRUE);
                Configuration::deleteByName('ACMB_SYNC_ERROR');
                foreach ($rows as $row) {
                    $i++;
                    $subscriber = json_decode($row['data'], true);
                    $merge_fields['email'] = $subscriber['email'];
                    array_push($list_subscriber, $merge_fields);
                    $list_id = $list_id . $row['id'];
                    if ($i < $total) {
                        $list_id = $list_id . ',';
                    }
                }
                $result = $this->api->batchAddSubscribers($client_subscription_list, $list_subscriber);

                $detail_error = Configuration::get('ACMB_SYNC_ERROR') ?? false;
                if ($result === false || $detail_error) {
                    $error = $this->l('Error de comunicación con el servidor');
                    $this->flush_sync_process();

                    die(json_encode([
                        'error' => $error,
                        'detail_error' => $detail_error
                    ]));
                } else { //success
                    Db::getInstance()->update(
                        'acmb_subscriptions',
                        ['synced' => 1],
                        'id IN ('. $list_id .')'
                    );
                    $processed++;
                }

                $remaining = Db::getInstance()->getValue("
                    SELECT COUNT(*) FROM "._DB_PREFIX_."acmb_subscriptions WHERE synced = 0
                ");

                if ($remaining < 1) {
                    $this->flush_sync_process();
                }

                die(json_encode([
                    'processed' => count($rows),
                    'hasMore' => ($remaining > 0)
                ]));
            } else {
                $this->flush_sync_process();
                $this->context->smarty->assign('sync_in_process', FALSE);
                if ($offset != 0 ) {
                    die(json_encode([
                        'hasMore' => FALSE
                    ]));
                } else {
                    die(json_encode(['error' => $this->l('No existen datos a sincronizar')]));
                }
            }

        } catch (Exception $e) {
            error_log($e->getMessage());
            Configuration::updateValue('ACMB_SYNC_ERROR',$e->getMessage());
            die(json_encode(['error' => $this->l('El proceso de sincronización ha sido detenido')]));
        }
    }

    private function flush_sync_process() {
        Configuration::deleteByName("ACMB_SYNC_IN_PROCESS");
        $this->dropTempDb();
    }

    private function createTempDb()
    {
        $this->dropTempDb();

        $sql = "
            CREATE TABLE IF NOT EXISTS `"._DB_PREFIX_."acmb_subscriptions` (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
                `email` VARCHAR(255) NOT NULL,
                `data` TEXT,
                `synced` TINYINT(1) DEFAULT 0,
                `date_added` DATETIME DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE (`email`)
            ) ENGINE="._MYSQL_ENGINE_." DEFAULT CHARSET=utf8;
        ";

        $result = False;
        try {
            $result = Db::getInstance()->execute($sql);
        } catch(Exception $e){
            error_log($e->getMessage());
        }
        return $result;
    }

    private function dropTempDb(){
        try {
            Db::getInstance()->execute('DROP TABLE IF EXISTS `'._DB_PREFIX_.'acmb_subscriptions`');
        } catch(Exception $e){
            error_log($e->getMessage());
        }
    }
}
